// Seed script to initialize database with sample data
import { MongoClient } from 'mongodb';
import { v4 as uuidv4 } from 'uuid';
import bcrypt from 'bcryptjs';

const MONGO_URL = process.env.MONGO_URL || 'mongodb://localhost:27017';

async function seed() {
  console.log('Connecting to MongoDB...');
  const client = await MongoClient.connect(MONGO_URL);
  const db = client.db('novafab_jewelry');

  console.log('Clearing existing data...');
  await db.collection('users').deleteMany({});
  await db.collection('categories').deleteMany({});
  await db.collection('subcategories').deleteMany({});
  await db.collection('products').deleteMany({});
  await db.collection('orders').deleteMany({});
  await db.collection('printLogs').deleteMany({});
  await db.collection('transactions').deleteMany({});
  await db.collection('machines').deleteMany({});

  // Create admin user
  console.log('Creating admin user...');
  const adminPassword = bcrypt.hashSync('admin123', 10);
  await db.collection('users').insertOne({
    id: uuidv4(),
    email: 'admin@novafab.com',
    password: adminPassword,
    name: 'Admin',
    company: 'NovaFab',
    role: 'admin',
    status: 'approved',
    language: 'fa',
    walletGrams: 1000,
    createdAt: new Date(),
    updatedAt: new Date()
  });

  // Create test user
  console.log('Creating test user...');
  const userPassword = bcrypt.hashSync('user123', 10);
  await db.collection('users').insertOne({
    id: uuidv4(),
    email: 'test@example.com',
    password: userPassword,
    name: 'Apa Ltd',
    company: 'Em0Jewelery',
    role: 'user',
    status: 'approved',
    language: 'fa',
    walletGrams: 1.38,
    createdAt: new Date(),
    updatedAt: new Date()
  });

  // Create machines
  console.log('Creating machines...');
  const machine1 = {
    id: uuidv4(),
    name: 'Mega 133 - 194',
    model: 'Mega 133',
    createdAt: new Date()
  };
  const machine2 = {
    id: uuidv4(),
    name: 'Mega 133 - 348',
    model: 'Mega 133',
    createdAt: new Date()
  };
  await db.collection('machines').insertMany([machine1, machine2]);

  // Create sample categories
  console.log('Creating categories...');
  const ringCategory = {
    id: uuidv4(),
    nameFA: 'انگشتر',
    nameEN: 'Ring',
    image: null,
    order: 1,
    createdAt: new Date()
  };
  
  const braceletCategory = {
    id: uuidv4(),
    nameFA: 'دستبند',
    nameEN: 'Bracelet',
    image: null,
    order: 2,
    createdAt: new Date()
  };

  await db.collection('categories').insertMany([ringCategory, braceletCategory]);

  // Create sample subcategories
  console.log('Creating subcategories...');
  const menRingSubcat = {
    id: uuidv4(),
    categoryId: ringCategory.id,
    nameFA: 'انگشتر مردانه',
    nameEN: 'Men Ring',
    image: null,
    averageWeight: 12.5,
    order: 1,
    createdAt: new Date()
  };

  const womenRingSubcat = {
    id: uuidv4(),
    categoryId: ringCategory.id,
    nameFA: 'انگشتر زنانه',
    nameEN: 'Women Ring',
    image: null,
    averageWeight: 8.3,
    order: 2,
    createdAt: new Date()
  };

  await db.collection('subcategories').insertMany([menRingSubcat, womenRingSubcat]);

  // Create sample products
  console.log('Creating products...');
  const product1 = {
    id: uuidv4(),
    subcategoryId: womenRingSubcat.id,
    code: 'BLV05',
    nameFA: 'دستبند طلا',
    nameEN: 'Gold Bracelet',
    images: [
      'https://via.placeholder.com/400x400/FFD700/000000?text=BLV05'
    ],
    weights: {
      '9k': 10.2,
      '14k': 15.6,
      '18k': 18.9,
      '21k': 21.5,
      '22k': 22.8
    },
    hasStone: false,
    variants: [
      { id: uuidv4(), size: 'B', weight: 0.35, karats: '14K' },
      { id: uuidv4(), size: 'BLV05-B-2', weight: 7.80, karats: '14K' },
      { id: uuidv4(), size: 'BLV05-B-1', weight: 7.44, karats: '14K' }
    ],
    createdAt: new Date()
  };

  await db.collection('products').insertOne(product1);

  console.log('✅ Seed completed successfully!');
  console.log('Admin credentials: admin@novafab.com / admin123');
  console.log('Test user credentials: test@example.com / user123');

  await client.close();
}

seed().catch(console.error);
