'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Card } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { format } from 'date-fns';

export default function OrdersManager({ locale }) {
  const [orders, setOrders] = useState([]);
  const [users, setUsers] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const token = localStorage.getItem('token');
      const [ordersRes, usersRes] = await Promise.all([
        fetch('/api/orders', { headers: { 'Authorization': `Bearer ${token}` } }),
        fetch('/api/users', { headers: { 'Authorization': `Bearer ${token}` } })
      ]);

      const ordersData = await ordersRes.json();
      const usersData = await usersRes.json();

      setOrders(ordersData.orders || []);
      setUsers(usersData.users || []);
    } catch (error) {
      console.error('Error loading data:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStatusChange = async (orderId, newStatus) => {
    try {
      const token = localStorage.getItem('token');
      await fetch(`/api/orders/${orderId}/status`, {
        method: 'PATCH',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ status: newStatus })
      });
      loadData();
    } catch (error) {
      console.error('Error updating order status:', error);
    }
  };

  const getStatusBadge = (status) => {
    const variants = {
      pending: 'secondary',
      payment_waiting: 'default',
      printing: 'default',
      ready: 'default',
      completed: 'default',
      cancelled: 'destructive'
    };

    const labels = {
      pending: 'در انتظار',
      payment_waiting: 'انتظار پرداخت',
      printing: 'در حال چاپ',
      ready: 'آماده',
      completed: 'تکمیل شده',
      cancelled: 'لغو شده'
    };

    return (
      <Badge variant={variants[status] || 'default'} className={
        status === 'ready' ? 'bg-green-500 text-white' :
        status === 'printing' ? 'bg-blue-500 text-white' :
        status === 'payment_waiting' ? 'bg-teal-500 text-white' : ''
      }>
        {labels[status] || status}
      </Badge>
    );
  };

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  return (
    <Card className="p-6">
      <div className="mb-4">
        <h2 className="text-2xl font-bold">مدیریت سفارش‌ها</h2>
      </div>

      <Table>
        <TableHeader>
          <TableRow>
            <TableHead>شماره سفارش</TableHead>
            <TableHead>کاربر</TableHead>
            <TableHead>تاریخ</TableHead>
            <TableHead>وزن طلا</TableHead>
            <TableHead>رزین</TableHead>
            <TableHead>وضعیت</TableHead>
            <TableHead>عملیات</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {orders.map((order) => {
            const user = users.find(u => u.id === order.userId);
            return (
              <TableRow key={order.id}>
                <TableCell className="font-medium">{order.orderNumber}</TableCell>
                <TableCell>
                  <div>
                    <div className="font-medium">{user?.name}</div>
                    <div className="text-xs text-muted-foreground">{user?.email}</div>
                  </div>
                </TableCell>
                <TableCell>{format(new Date(order.createdAt), 'yyyy-MM-dd')}</TableCell>
                <TableCell>{order.totalGoldGrams?.toFixed(2)} گرم</TableCell>
                <TableCell>{order.totalResinCredit?.toFixed(2)}</TableCell>
                <TableCell>{getStatusBadge(order.status)}</TableCell>
                <TableCell>
                  <Select
                    value={order.status}
                    onValueChange={(value) => handleStatusChange(order.id, value)}
                  >
                    <SelectTrigger className="w-40">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="pending">در انتظار</SelectItem>
                      <SelectItem value="payment_waiting">انتظار پرداخت</SelectItem>
                      <SelectItem value="printing">در حال چاپ</SelectItem>
                      <SelectItem value="ready">آماده</SelectItem>
                      <SelectItem value="completed">تکمیل شده</SelectItem>
                      <SelectItem value="cancelled">لغو شده</SelectItem>
                    </SelectContent>
                  </Select>
                </TableCell>
              </TableRow>
            );
          })}
        </TableBody>
      </Table>
    </Card>
  );
}