'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Trash2, Plus, Edit } from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';
import ImageUploader from '@/components/ImageUploader';
import { Separator } from '@/components/ui/separator';

export default function EnhancedProductsManager({ locale }) {
  const [products, setProducts] = useState([]);
  const [categories, setCategories] = useState([]);
  const [subcategories, setSubcategories] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [message, setMessage] = useState({ type: '', text: '' });
  const [formData, setFormData] = useState({
    subcategoryId: '',
    code: '',
    nameFA: '',
    nameEN: '',
    images: [],
    weights: {
      '9k': '',
      '14k': '',
      '18k': '',
      '21k': '',
      '22k': ''
    },
    hasStone: false,
    variants: []
  });
  const [newVariant, setNewVariant] = useState({ size: '', weight: '', karats: '14K' });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [productsRes, categoriesRes, subcategoriesRes] = await Promise.all([
        fetch('/api/products'),
        fetch('/api/categories'),
        fetch('/api/subcategories')
      ]);

      const productsData = await productsRes.json();
      const categoriesData = await categoriesRes.json();
      const subcategoriesData = await subcategoriesRes.json();

      setProducts(productsData.products || []);
      setCategories(categoriesData.categories || []);
      setSubcategories(subcategoriesData.subcategories || []);
    } catch (error) {
      console.error('Error loading data:', error);
      setMessage({ type: 'error', text: 'خطا در بارگذاری اطلاعات' });
    } finally {
      setLoading(false);
    }
  };

  const handleSave = async () => {
    try {
      if (!formData.subcategoryId || !formData.nameFA || !formData.nameEN) {
        setMessage({ type: 'error', text: 'لطفاً فیلدهای الزامی را پر کنید' });
        return;
      }

      const token = localStorage.getItem('token');

      // Convert weights to numbers
      const weights = {};
      Object.keys(formData.weights).forEach(karat => {
        const value = parseFloat(formData.weights[karat]);
        if (value > 0) weights[karat] = value;
      });

      const response = await fetch('/api/products', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          subcategoryId: formData.subcategoryId,
          code: formData.code,
          nameFA: formData.nameFA,
          nameEN: formData.nameEN,
          images: formData.images,
          weights,
          hasStone: formData.hasStone,
          variants: formData.variants
        })
      });

      if (response.ok) {
        setMessage({ type: 'success', text: 'محصول با موفقیت اضافه شد' });
        setShowModal(false);
        resetForm();
        loadData();
        setTimeout(() => setMessage({ type: '', text: '' }), 3000);
      } else {
        throw new Error('خطا در ذخیره‌سازی');
      }
    } catch (error) {
      console.error('Error saving product:', error);
      setMessage({ type: 'error', text: 'خطا در ذخیره‌سازی محصول' });
    }
  };

  const handleDelete = async (id) => {
    if (!confirm('آیا مطمئن هستید؟')) return;

    try {
      const token = localStorage.getItem('token');
      const response = await fetch(`/api/products/${id}`, {
        method: 'DELETE',
        headers: { 'Authorization': `Bearer ${token}` }
      });

      if (response.ok) {
        setMessage({ type: 'success', text: 'محصول با موفقیت حذف شد' });
        loadData();
        setTimeout(() => setMessage({ type: '', text: '' }), 3000);
      }
    } catch (error) {
      console.error('Error deleting product:', error);
      setMessage({ type: 'error', text: 'خطا در حذف محصول' });
    }
  };

  const addVariant = () => {
    if (!newVariant.size || !newVariant.weight) {
      setMessage({ type: 'error', text: 'لطفاً سایز و وزن را وارد کنید' });
      return;
    }

    setFormData({
      ...formData,
      variants: [...formData.variants, { ...newVariant, weight: parseFloat(newVariant.weight) }]
    });
    setNewVariant({ size: '', weight: '', karats: '14K' });
  };

  const removeVariant = (index) => {
    setFormData({
      ...formData,
      variants: formData.variants.filter((_, i) => i !== index)
    });
  };

  const resetForm = () => {
    setFormData({
      subcategoryId: '',
      code: '',
      nameFA: '',
      nameEN: '',
      images: [],
      weights: { '9k': '', '14k': '', '18k': '', '21k': '', '22k': '' },
      hasStone: false,
      variants: []
    });
    setNewVariant({ size: '', weight: '', karats: '14K' });
  };

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  return (
    <div className="space-y-6">
      {message.text && (
        <Alert variant={message.type === 'error' ? 'destructive' : 'default'}>
          <AlertDescription>{message.text}</AlertDescription>
        </Alert>
      )}

      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-2xl font-bold">مدیریت محصولات</h2>
          <Button onClick={() => setShowModal(true)}>
            <Plus className="h-4 w-4 mr-2" />
            افزودن محصول جدید
          </Button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          {products.map((product) => {
            const subcategory = subcategories.find(s => s.id === product.subcategoryId);
            const category = categories.find(c => 
              subcategories.some(sub => sub.id === product.subcategoryId && sub.categoryId === c.id)
            );
            
            return (
              <Card key={product.id} className="p-4 border-2 hover:border-primary transition-colors">
                {product.images && product.images[0] && (
                  <div className="aspect-square bg-muted rounded-lg mb-3 overflow-hidden">
                    <img 
                      src={product.images[0]} 
                      alt={product.nameFA} 
                      className="w-full h-full object-cover"
                    />
                  </div>
                )}
                {product.images && product.images.length > 1 && (
                  <div className="grid grid-cols-3 gap-1 mb-3">
                    {product.images.slice(1, 4).map((img, idx) => (
                      <div key={idx} className="aspect-square bg-muted rounded overflow-hidden">
                        <img src={img} alt="" className="w-full h-full object-cover" />
                      </div>
                    ))}
                  </div>
                )}
                
                <div className="space-y-2">
                  <div>
                    <div className="font-bold text-lg">{product.nameFA}</div>
                    <div className="text-sm text-muted-foreground">{product.nameEN}</div>
                  </div>
                  
                  {product.code && (
                    <div className="text-xs font-mono bg-muted px-2 py-1 rounded">
                      {product.code}
                    </div>
                  )}
                  
                  <div className="text-xs text-muted-foreground">
                    {category?.nameFA} › {subcategory?.nameFA}
                  </div>
                  
                  <div className="flex flex-wrap gap-1">
                    {product.hasStone && (
                      <span className="text-xs bg-amber-100 text-amber-800 px-2 py-1 rounded">
                        💎 سنگ‌دار
                      </span>
                    )}
                    {product.variants && product.variants.length > 0 && (
                      <span className="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded">
                        📏 {product.variants.length} سایز
                      </span>
                    )}
                    {product.weights && Object.keys(product.weights).length > 0 && (
                      <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                        ⚖️ {Object.keys(product.weights).length} عیار
                      </span>
                    )}
                  </div>
                  
                  {product.variants && product.variants.length > 0 && (
                    <div className="text-xs space-y-1 bg-muted p-2 rounded">
                      <div className="font-semibold">سایزها:</div>
                      {product.variants.slice(0, 3).map((v, i) => (
                        <div key={i} className="flex justify-between">
                          <span>{v.size}</span>
                          <span>{v.weight}g ({v.karats})</span>
                        </div>
                      ))}
                      {product.variants.length > 3 && (
                        <div className="text-center text-muted-foreground">
                          +{product.variants.length - 3} مورد دیگر
                        </div>
                      )}
                    </div>
                  )}
                  
                  <Button
                    size="sm"
                    variant="destructive"
                    className="w-full"
                    onClick={() => handleDelete(product.id)}
                  >
                    <Trash2 className="h-3 w-3 mr-2" />
                    حذف
                  </Button>
                </div>
              </Card>
            );
          })}
        </div>

        {products.length === 0 && (
          <div className="text-center py-12 text-muted-foreground">
            <p>هیچ محصولی وجود ندارد</p>
            <p className="text-sm mt-2">برای شروع، یک محصول جدید اضافه کنید</p>
          </div>
        )}
      </Card>

      {/* Add Product Modal */}
      <Dialog open={showModal} onOpenChange={(open) => {
        setShowModal(open);
        if (!open) resetForm();
      }}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>افزودن محصول جدید</DialogTitle>
          </DialogHeader>
          
          <div className="space-y-6 py-4">
            {/* Basic Info */}
            <div className="space-y-4">
              <h3 className="font-semibold text-lg">اطلاعات پایه</h3>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label>زیردسته *</Label>
                  <Select value={formData.subcategoryId} onValueChange={(value) => setFormData({ ...formData, subcategoryId: value })}>
                    <SelectTrigger>
                      <SelectValue placeholder="انتخاب زیردسته" />
                    </SelectTrigger>
                    <SelectContent>
                      {subcategories.map(sub => {
                        const cat = categories.find(c => c.id === sub.categoryId);
                        return (
                          <SelectItem key={sub.id} value={sub.id}>
                            {cat?.nameFA} › {sub.nameFA}
                          </SelectItem>
                        );
                      })}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label>کد محصول</Label>
                  <Input
                    value={formData.code}
                    onChange={(e) => setFormData({ ...formData, code: e.target.value })}
                    placeholder="BLV05"
                  />
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label>نام فارسی *</Label>
                  <Input
                    value={formData.nameFA}
                    onChange={(e) => setFormData({ ...formData, nameFA: e.target.value })}
                    placeholder="دستبند طلا"
                  />
                </div>

                <div>
                  <Label>نام انگلیسی *</Label>
                  <Input
                    value={formData.nameEN}
                    onChange={(e) => setFormData({ ...formData, nameEN: e.target.value })}
                    placeholder="Gold Bracelet"
                  />
                </div>
              </div>

              <div className="flex items-center space-x-2 space-x-reverse">
                <Checkbox
                  id="hasStone"
                  checked={formData.hasStone}
                  onCheckedChange={(checked) => setFormData({ ...formData, hasStone: checked })}
                />
                <Label htmlFor="hasStone" className="cursor-pointer">
                  محصول سنگ‌دار است
                </Label>
              </div>
            </div>

            <Separator />

            {/* Images */}
            <div>
              <h3 className="font-semibold text-lg mb-4">تصاویر محصول</h3>
              <ImageUploader
                images={formData.images}
                onChange={(images) => setFormData({ ...formData, images })}
                maxImages={5}
                label="تصاویر (حداکثر 5 عکس)"
              />
            </div>

            <Separator />

            {/* Weights by Karat */}
            <div>
              <h3 className="font-semibold text-lg mb-4">وزن بر اساس عیار (گرم)</h3>
              <div className="grid grid-cols-5 gap-3">
                {['9k', '14k', '18k', '21k', '22k'].map(karat => (
                  <div key={karat}>
                    <Label className="text-xs font-bold uppercase">{karat}</Label>
                    <Input
                      type="number"
                      value={formData.weights[karat]}
                      onChange={(e) => setFormData({
                        ...formData,
                        weights: { ...formData.weights, [karat]: e.target.value }
                      })}
                      placeholder="10.5"
                      step="0.01"
                    />
                  </div>
                ))}
              </div>
              <p className="text-xs text-muted-foreground mt-2">
                💡 فقط عیارهایی که وزن دارند را پر کنید
              </p>
            </div>

            <Separator />

            {/* Variants (Sizes) */}
            <div>
              <h3 className="font-semibold text-lg mb-4">سایزها و وزن‌های مختلف</h3>
              
              {formData.variants.length > 0 && (
                <div className="space-y-2 mb-4">
                  {formData.variants.map((variant, index) => (
                    <div key={index} className="flex items-center gap-2 bg-muted p-3 rounded-lg">
                      <div className="flex-1 grid grid-cols-3 gap-2 text-sm">
                        <div>
                          <span className="text-muted-foreground">سایز:</span>
                          <span className="font-medium ml-2">{variant.size}</span>
                        </div>
                        <div>
                          <span className="text-muted-foreground">وزن:</span>
                          <span className="font-medium ml-2">{variant.weight} گرم</span>
                        </div>
                        <div>
                          <span className="text-muted-foreground">عیار:</span>
                          <span className="font-medium ml-2">{variant.karats}</span>
                        </div>
                      </div>
                      <Button 
                        size="sm" 
                        variant="ghost" 
                        onClick={() => removeVariant(index)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                </div>
              )}
              
              <div className="grid grid-cols-4 gap-2">
                <Input
                  placeholder="سایز (مثلا B)"
                  value={newVariant.size}
                  onChange={(e) => setNewVariant({ ...newVariant, size: e.target.value })}
                />
                <Input
                  type="number"
                  placeholder="وزن (گرم)"
                  value={newVariant.weight}
                  onChange={(e) => setNewVariant({ ...newVariant, weight: e.target.value })}
                  step="0.01"
                />
                <Select value={newVariant.karats} onValueChange={(value) => setNewVariant({ ...newVariant, karats: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="9K">9K</SelectItem>
                    <SelectItem value="14K">14K</SelectItem>
                    <SelectItem value="18K">18K</SelectItem>
                    <SelectItem value="21K">21K</SelectItem>
                    <SelectItem value="22K">22K</SelectItem>
                  </SelectContent>
                </Select>
                <Button onClick={addVariant}>
                  <Plus className="h-4 w-4 mr-2" />
                  افزودن
                </Button>
              </div>
              <p className="text-xs text-muted-foreground mt-2">
                💡 سایزها برای محصولاتی که در اندازه‌های مختلف تولید می‌شوند
              </p>
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowModal(false);
              resetForm();
            }}>
              انصراف
            </Button>
            <Button onClick={handleSave}>
              ذخیره محصول
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
