'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Trash2, Edit, Plus, Image as ImageIcon } from 'lucide-react';
import ImageUploader from '@/components/ImageUploader';
import { Alert, AlertDescription } from '@/components/ui/alert';

export default function EnhancedCategoriesManager({ locale }) {
  const [categories, setCategories] = useState([]);
  const [subcategories, setSubcategories] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [modalType, setModalType] = useState('category');
  const [selectedCategory, setSelectedCategory] = useState(null);
  const [message, setMessage] = useState({ type: '', text: '' });
  const [formData, setFormData] = useState({
    nameFA: '',
    nameEN: '',
    images: [],
    averageWeight: 0,
    order: 0
  });

  useEffect(() => {
    loadCategories();
    loadSubcategories();
  }, []);

  const loadCategories = async () => {
    try {
      const response = await fetch('/api/categories');
      const data = await response.json();
      setCategories(data.categories || []);
    } catch (error) {
      console.error('Error loading categories:', error);
      setMessage({ type: 'error', text: 'خطا در بارگذاری دسته‌بندی‌ها' });
    } finally {
      setLoading(false);
    }
  };

  const loadSubcategories = async () => {
    try {
      const response = await fetch('/api/subcategories');
      const data = await response.json();
      setSubcategories(data.subcategories || []);
    } catch (error) {
      console.error('Error loading subcategories:', error);
    }
  };

  const handleSave = async () => {
    try {
      if (!formData.nameFA || !formData.nameEN) {
        setMessage({ type: 'error', text: 'لطفاً نام فارسی و انگلیسی را وارد کنید' });
        return;
      }

      const token = localStorage.getItem('token');
      const endpoint = modalType === 'category' ? '/api/categories' : '/api/subcategories';
      
      const body = modalType === 'category' ? {
        nameFA: formData.nameFA,
        nameEN: formData.nameEN,
        image: formData.images[0] || null,
        order: parseInt(formData.order) || 0
      } : {
        categoryId: selectedCategory,
        nameFA: formData.nameFA,
        nameEN: formData.nameEN,
        image: formData.images[0] || null,
        averageWeight: parseFloat(formData.averageWeight) || 0,
        order: parseInt(formData.order) || 0
      };

      const response = await fetch(endpoint, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(body)
      });

      if (response.ok) {
        setMessage({ type: 'success', text: 'با موفقیت ذخیره شد' });
        setShowModal(false);
        resetForm();
        loadCategories();
        loadSubcategories();
        setTimeout(() => setMessage({ type: '', text: '' }), 3000);
      } else {
        throw new Error('خطا در ذخیره‌سازی');
      }
    } catch (error) {
      console.error('Error saving:', error);
      setMessage({ type: 'error', text: 'خطا در ذخیره‌سازی' });
    }
  };

  const handleDelete = async (id, type) => {
    if (!confirm('آیا مطمئن هستید؟')) return;

    try {
      const token = localStorage.getItem('token');
      const endpoint = type === 'category' ? `/api/categories/${id}` : `/api/subcategories/${id}`;
      
      const response = await fetch(endpoint, {
        method: 'DELETE',
        headers: { 'Authorization': `Bearer ${token}` }
      });

      if (response.ok) {
        setMessage({ type: 'success', text: 'با موفقیت حذف شد' });
        loadCategories();
        loadSubcategories();
        setTimeout(() => setMessage({ type: '', text: '' }), 3000);
      }
    } catch (error) {
      console.error('Error deleting:', error);
      setMessage({ type: 'error', text: 'خطا در حذف' });
    }
  };

  const resetForm = () => {
    setFormData({
      nameFA: '',
      nameEN: '',
      images: [],
      averageWeight: 0,
      order: 0
    });
    setSelectedCategory(null);
  };

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  return (
    <div className="space-y-6">
      {message.text && (
        <Alert variant={message.type === 'error' ? 'destructive' : 'default'}>
          <AlertDescription>{message.text}</AlertDescription>
        </Alert>
      )}

      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-2xl font-bold">مدیریت دسته‌بندی‌ها و زیردسته‌ها</h2>
          <Button onClick={() => {
            setModalType('category');
            setShowModal(true);
          }}>
            <Plus className="h-4 w-4 mr-2" />
            افزودن دسته‌بندی جدید
          </Button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {categories.map((cat) => (
            <Card key={cat.id} className="p-4 border-2">
              {/* Category Image */}
              {cat.image && (
                <div className="aspect-video bg-muted rounded-lg overflow-hidden mb-3">
                  <img
                    src={cat.image}
                    alt={cat.nameFA}
                    className="w-full h-full object-cover"
                  />
                </div>
              )}
              
              <div className="flex items-start justify-between mb-3">
                <div className="flex-1">
                  <h3 className="font-bold text-lg">{cat.nameFA}</h3>
                  <p className="text-sm text-muted-foreground">{cat.nameEN}</p>
                  <p className="text-xs text-muted-foreground mt-1">ترتیب: {cat.order}</p>
                </div>
                <Button
                  size="sm"
                  variant="destructive"
                  onClick={() => handleDelete(cat.id, 'category')}
                >
                  <Trash2 className="h-4 w-4" />
                </Button>
              </div>
              
              <div className="border-t pt-3 mb-3">
                <Button
                  size="sm"
                  variant="outline"
                  className="w-full"
                  onClick={() => {
                    setModalType('subcategory');
                    setSelectedCategory(cat.id);
                    setShowModal(true);
                  }}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  افزودن زیردسته
                </Button>
              </div>

              {/* Subcategories */}
              <div className="space-y-2">
                <p className="text-sm font-semibold text-muted-foreground mb-2">
                  زیردسته‌ها ({subcategories.filter(sub => sub.categoryId === cat.id).length})
                </p>
                {subcategories
                  .filter(sub => sub.categoryId === cat.id)
                  .map(sub => (
                    <div key={sub.id} className="bg-muted p-3 rounded-lg">
                      {sub.image && (
                        <div className="aspect-video bg-background rounded overflow-hidden mb-2">
                          <img
                            src={sub.image}
                            alt={sub.nameFA}
                            className="w-full h-full object-cover"
                          />
                        </div>
                      )}
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <div className="font-medium text-sm">{sub.nameFA}</div>
                          <div className="text-xs text-muted-foreground">{sub.nameEN}</div>
                          <div className="text-xs text-muted-foreground mt-1">
                            میانگین وزن: {sub.averageWeight} گرم | ترتیب: {sub.order}
                          </div>
                        </div>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={() => handleDelete(sub.id, 'subcategory')}
                        >
                          <Trash2 className="h-3 w-3" />
                        </Button>
                      </div>
                    </div>
                  ))}
              </div>
            </Card>
          ))}
        </div>
      </Card>

      {/* Add/Edit Modal */}
      <Dialog open={showModal} onOpenChange={(open) => {
        setShowModal(open);
        if (!open) resetForm();
      }}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {modalType === 'category' ? 'افزودن دسته‌بندی جدید' : 'افزودن زیردسته جدید'}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label>نام فارسی *</Label>
                <Input
                  value={formData.nameFA}
                  onChange={(e) => setFormData({ ...formData, nameFA: e.target.value })}
                  placeholder="مثال: انگشتر"
                />
              </div>

              <div>
                <Label>نام انگلیسی *</Label>
                <Input
                  value={formData.nameEN}
                  onChange={(e) => setFormData({ ...formData, nameEN: e.target.value })}
                  placeholder="Example: Ring"
                />
              </div>
            </div>

            {modalType === 'subcategory' && (
              <div>
                <Label>میانگین وزن (گرم)</Label>
                <Input
                  type="number"
                  value={formData.averageWeight}
                  onChange={(e) => setFormData({ ...formData, averageWeight: e.target.value })}
                  placeholder="12.5"
                  step="0.1"
                />
              </div>
            )}

            <div>
              <Label>ترتیب نمایش</Label>
              <Input
                type="number"
                value={formData.order}
                onChange={(e) => setFormData({ ...formData, order: e.target.value })}
                placeholder="1"
              />
            </div>

            <ImageUploader
              images={formData.images}
              onChange={(images) => setFormData({ ...formData, images })}
              maxImages={1}
              label={`تصویر ${modalType === 'category' ? 'دسته‌بندی' : 'زیردسته'} (اختیاری)`}
            />
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowModal(false);
              resetForm();
            }}>
              انصراف
            </Button>
            <Button onClick={handleSave}>
              ذخیره
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}